/*
<samplecode>
  <abstract>
Extension to the saved-game class to generate random bytes, simulating a larger saved-game file.
  </abstract>
</samplecode>
*/

import Foundation
import Security
import os.log

fileprivate extension Logger {
    static let SaveGameSecurity = Logger(subsystem: Self.loggingSubsystem, category: "SaveGame+Security")
}

extension SaveGame {
    static func generateRandomBytes(_ measurement: Measurement<UnitInformationStorage>) throws -> Data {
        let byteCount = Int(measurement.converted(to: .bytes).value)
        return try generateRandomBytes(byteCount)
    }
    
    private static func generateRandomBytes(_ count: Int) throws -> Data {
        var randomBytes = Data(count: count)
        let status = randomBytes.withUnsafeMutableBytes { (pointer: UnsafeMutableRawBufferPointer) -> OSStatus in
            guard let baseAddress = pointer.baseAddress else { return errSecInvalidPointer }
            return SecRandomCopyBytes(kSecRandomDefault, count, baseAddress)
        }

        if status == errSecSuccess { // Test the status.
            return randomBytes
        } else {
            Logger.SaveGameSecurity.error("Can't generate random bytes.")
            throw SaveGameError.randomBytesGenerationFailed
        }
    }
    
    enum SaveGameError: Error {
        case randomBytesGenerationFailed
    }
}
